use std::fmt::{self, Write};

use comrak::html::{render_sourcepos, ChildRendering, Context};
use comrak::nodes::{AstNode, ListType, NodeValue};
use comrak::{create_formatter, html};
use lazy_static::lazy_static;
use regex::Regex;

// TODO: Use std::sync:LazyLock once we're on 1.80+.
// https://doc.rust-lang.org/std/sync/struct.LazyLock.html
lazy_static! {
    static ref PLACEHOLDER_REGEX: Regex = Regex::new(r"%(\{|%7B)(\w{1,30})(}|%7D)").unwrap();
}

pub struct RenderUserData {
    pub default_html: bool,
    pub inapplicable_tasks: bool,
    pub placeholder_detection: bool,
    pub only_escape_chars: Option<Vec<char>>,
    pub debug: bool,
}

// The important thing to remember is that this overrides the default behavior of the
// specified nodes. If we do override a node, then it's our responsibility to ensure that
// any changes in the `comrak` code for those nodes is backported to here, such as when
// `figcaption` support was added.
// One idea to limit that would be having the ability to specify attributes that would
// be inserted when a node is rendered. That would allow us to (in many cases) just
// inject the changes we need. Such a feature would need to be added to `comrak`.
create_formatter!(CustomFormatter<RenderUserData>, {
    NodeValue::Text(_) => |context, node, entering| {
        return render_text(context, node, entering);
    },
    NodeValue::Link(_) => |context, node, entering| {
        return render_link(context, node, entering);
    },
    NodeValue::Image(_) => |context, node, entering| {
        return render_image(context, node, entering);
    },
    NodeValue::List(_) => |context, node, entering| {
        return render_list(context, node, entering);
    },
    NodeValue::TaskItem(_) => |context, node, entering| {
        return render_task_item(context, node, entering);
    },
    NodeValue::Escaped => |context, node, entering| {
        return render_escaped(context, node, entering);
    },
});

fn render_image<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
) -> Result<ChildRendering, fmt::Error> {
    let NodeValue::Image(ref nl) = node.data.borrow().value else {
        unreachable!()
    };

    if !(context.user.placeholder_detection && PLACEHOLDER_REGEX.is_match(&nl.url)) {
        return html::format_node_default(context, node, entering);
    }

    if entering {
        if context.options.render.figure_with_caption {
            context.write_str("<figure>")?;
        }
        context.write_str("<img")?;
        html::render_sourcepos(context, node)?;
        context.write_str(" src=\"")?;
        if context.options.render.unsafe_ || !html::dangerous_url(&nl.url) {
            if let Some(rewriter) = &context.options.extension.image_url_rewriter {
                context.escape_href(&rewriter.to_html(&nl.url))?;
            } else {
                context.escape_href(&nl.url)?;
            }
        }

        context.write_str("\"")?;

        if PLACEHOLDER_REGEX.is_match(&nl.url) {
            context.write_str(" data-placeholder")?;
        }

        context.write_str(" alt=\"")?;

        return Ok(ChildRendering::Plain);
    } else {
        if !nl.title.is_empty() {
            context.write_str("\" title=\"")?;
            context.escape(&nl.title)?;
        }
        context.write_str("\" />")?;
        if context.options.render.figure_with_caption {
            if !nl.title.is_empty() {
                context.write_str("<figcaption>")?;
                context.escape(&nl.title)?;
                context.write_str("</figcaption>")?;
            }
            context.write_str("</figure>")?;
        };
    }

    Ok(ChildRendering::HTML)
}

fn render_link<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
) -> Result<ChildRendering, fmt::Error> {
    let NodeValue::Link(ref nl) = node.data.borrow().value else {
        unreachable!()
    };

    if !(context.user.placeholder_detection && PLACEHOLDER_REGEX.is_match(&nl.url)) {
        return html::format_node_default(context, node, entering);
    }

    let parent_node = node.parent();

    if !context.options.parse.relaxed_autolinks
        || (parent_node.is_none()
            || !matches!(
                parent_node.unwrap().data.borrow().value,
                NodeValue::Link(..)
            ))
    {
        if entering {
            context.write_str("<a")?;
            html::render_sourcepos(context, node)?;
            context.write_str(" href=\"")?;
            if context.options.render.unsafe_ || !html::dangerous_url(&nl.url) {
                if let Some(rewriter) = &context.options.extension.link_url_rewriter {
                    context.escape_href(&rewriter.to_html(&nl.url))?;
                } else {
                    context.escape_href(&nl.url)?;
                }
            }
            context.write_str("\"")?;

            if !nl.title.is_empty() {
                context.write_str(" title=\"")?;
                context.escape(&nl.title)?;
            }

            if PLACEHOLDER_REGEX.is_match(&nl.url) {
                context.write_str(" data-placeholder")?;
            }

            context.write_str(">")?;
        } else {
            context.write_str("</a>")?;
        }
    }

    Ok(ChildRendering::HTML)
}

// Overridden to use class `task-list` instead of `contains-task-list`
// to align with GitLab class usage
fn render_list<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
) -> Result<ChildRendering, fmt::Error> {
    if !entering || !context.options.render.tasklist_classes {
        return html::format_node_default(context, node, entering);
    }

    let NodeValue::List(ref nl) = node.data.borrow().value else {
        unreachable!()
    };

    context.cr()?;
    match nl.list_type {
        ListType::Bullet => {
            context.write_str("<ul")?;
            if nl.is_task_list {
                context.write_str(" class=\"task-list\"")?;
            }
            html::render_sourcepos(context, node)?;
            context.write_str(">\n")?;
        }
        ListType::Ordered => {
            context.write_str("<ol")?;
            if nl.is_task_list {
                context.write_str(" class=\"task-list\"")?;
            }
            html::render_sourcepos(context, node)?;
            if nl.start == 1 {
                context.write_str(">\n")?;
            } else {
                writeln!(context, " start=\"{}\">", nl.start)?;
            }
        }
    }

    Ok(ChildRendering::HTML)
}

// Overridden to detect inapplicable task list items
fn render_task_item<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
) -> Result<ChildRendering, fmt::Error> {
    if !context.user.inapplicable_tasks {
        return html::format_node_default(context, node, entering);
    }

    let NodeValue::TaskItem(symbol) = node.data.borrow().value else {
        unreachable!()
    };

    if symbol.is_none() || matches!(symbol, Some('x' | 'X')) {
        return html::format_node_default(context, node, entering);
    }

    if entering {
        // Handle an inapplicable task symbol.
        if matches!(symbol, Some('~')) {
            context.cr()?;
            context.write_str("<li")?;
            context.write_str(" class=\"inapplicable")?;

            if context.options.render.tasklist_classes {
                context.write_str(" task-list-item")?;
            }
            context.write_str("\"")?;

            html::render_sourcepos(context, node)?;
            context.write_str(">")?;
            context.write_str("<input type=\"checkbox\"")?;

            if context.options.render.tasklist_classes {
                context.write_str(" class=\"task-list-item-checkbox\"")?;
            }

            context.write_str(" data-inapplicable disabled=\"\"> ")?;
        } else {
            // Don't allow unsupported symbols to render a checkbox
            context.cr()?;
            context.write_str("<li")?;

            if context.options.render.tasklist_classes {
                context.write_str(" class=\"task-list-item\"")?;
            }

            html::render_sourcepos(context, node)?;
            context.write_str(">")?;
            context.write_str("[")?;
            context.escape(&symbol.unwrap().to_string())?;
            context.write_str("] ")?;
        }
    } else {
        context.write_str("</li>\n")?;
    }

    Ok(ChildRendering::HTML)
}

fn render_text<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
) -> Result<ChildRendering, fmt::Error> {
    let NodeValue::Text(ref literal) = node.data.borrow().value else {
        unreachable!()
    };

    if !(context.user.placeholder_detection && PLACEHOLDER_REGEX.is_match(literal)) {
        return html::format_node_default(context, node, entering);
    }

    // Don't currently support placeholders in the text inside links or images.
    // If the text has an underscore in it, then the parser will not combine
    // the multiple text nodes in `comrak`'s `postprocess_text_nodes`, breaking up
    // the placeholder into multiple text nodes.
    // For example, `[%{a_b}](link)`.
    let parent = node.parent().unwrap();
    if matches!(
        parent.data.borrow().value,
        NodeValue::Link(_) | NodeValue::Image(_)
    ) {
        return html::format_node_default(context, node, entering);
    }

    if entering {
        let mut cursor: usize = 0;

        for mat in PLACEHOLDER_REGEX.find_iter(literal) {
            if mat.start() > cursor {
                context.escape(&literal[cursor..mat.start()])?;
            }

            context.write_str("<span data-placeholder>")?;
            context.escape(&literal[mat.start()..mat.end()])?;
            context.write_str("</span>")?;

            cursor = mat.end();
        }

        if cursor < literal.len() {
            context.escape(&literal[cursor..literal.len()])?;
        }
    }

    Ok(ChildRendering::HTML)
}

fn render_escaped<'a>(
    context: &mut Context<RenderUserData>,
    node: &'a AstNode<'a>,
    entering: bool,
) -> Result<ChildRendering, fmt::Error> {
    if !context.options.render.escaped_char_spans {
        return Ok(ChildRendering::HTML);
    }

    if context.user.only_escape_chars.is_none()
        || with_node_text_content(node, false, |content| {
            if content.len() != 1 {
                return false;
            }
            let c = content.chars().next().unwrap();
            context
                .user
                .only_escape_chars
                .as_ref()
                .unwrap()
                .contains(&c)
        })
    {
        if entering {
            context.write_str("<span data-escaped-char")?;
            render_sourcepos(context, node)?;
            context.write_str(">")?;
        } else {
            context.write_str("</span>")?;
        }
    }

    Ok(ChildRendering::HTML)
}

/// If the given node has a single text child, apply a function to the text content
/// of that node. Otherwise, return the given default value.
fn with_node_text_content<'a, U, F>(node: &'a AstNode<'a>, default: U, f: F) -> U
where
    F: FnOnce(&str) -> U,
{
    let Some(child) = node.first_child() else {
        return default;
    };
    let Some(last_child) = node.last_child() else {
        return default;
    };
    if !child.same_node(last_child) {
        return default;
    }
    let NodeValue::Text(ref text) = child.data.borrow().value else {
        return default;
    };
    f(text)
}
