/* GCompris - guesscount.qml
 *
 * SPDX-FileCopyrightText: 2016 RAHUL YADAV <rahulyadav170923@gmail.com>
 *
 * Authors:
 *   Pascal Georges <pascal.georges1@free.fr> (GTK+ version)
 *   RAHUL YADAV <rahulyadav170923@gmail.com> (Qt Quick port)
 *
 *   SPDX-License-Identifier: GPL-3.0-or-later
 */
import QtQuick 2.12
import GCompris 1.0
import "../../core"
import "guesscount.js" as Activity
import "dataset.js" as Data

ActivityBase {
    id: activity

    onStart: focus = true
    onStop: {}
    property bool needRestart: true

    pageComponent: Image {
        id: background
        anchors.fill: parent
        source: Activity.baseUrl + "/backgroundW01.svg"
        signal start
        signal stop
        readonly property int baseMargins: 10 * ApplicationInfo.ratio
        readonly property int halfMargins: 5 * ApplicationInfo.ratio
        readonly property double tileRadius: 6 * ApplicationInfo.ratio
        readonly property double tileBorder: 3 * ApplicationInfo.ratio

        Component.onCompleted: {
            dialogActivityConfig.initialize()
            activity.start.connect(start)
            activity.stop.connect(stop)
        }
        MouseArea {
            anchors.fill: parent
            onClicked: {
                if(warningDialog.visible)
                    warningDialog.visible = false
            }
        }

        // Add here the QML items you need to access in javascript
        QtObject {
            id: items
            property Item main: activity.main
            property alias background: background
            property int currentLevel: activity.currentLevel
            property alias bonus: bonus
            property alias score: score
            property alias operatorRow: operatorRow
            property alias operandRow: operandRow
            property var data
            readonly property var levels: activity.datasets
            property int result: 0
            property alias warningDialog: warningDialog
            property alias goodAnswerSound: goodAnswerSound
            property bool clearOperations: false
            property bool solved: false
            property var levelArr: Data.defaultOperators
            property string mode: "builtin"
            property int currentlevel
        }

        onStart: if (activity.needRestart) {
                     Activity.start(items);
                     activity.needRestart = false;
                 }
                 else
                     Activity.initLevel();
        onStop: { Activity.stop() }


        GCSoundEffect {
            id: goodAnswerSound
            source: "qrc:/gcompris/src/core/resource/sounds/completetask.wav"
        }

        JsonParser {
            id: parser
            onError: (msg) => console.error("Guesscount: Error parsing JSON: " + msg);
        }

        Loader {
            id: admin
            active: false
            sourceComponent: Column {
                spacing: background.baseMargins
                width: parent.width
                height: parent.height

                Repeater {
                    id:levels
                    model: 8
                    Admin {
                        id:level
                        level: modelData+1
                        width: parent.width
                        height: parent.height
                        data: items.data
                    }
                }
            }
        }

        Item {
            id: top
            z: 10
            height: score.height
            anchors {
                left: parent.left
                right: parent.right
                top: parent.top
                margins: background.baseMargins
            }
            Score {
                id: score
                anchors.left: parent.left
                anchors.right: undefined
                anchors.top: undefined
                anchors.bottom: undefined
                anchors.margins: 0
                anchors.verticalCenter: parent.verticalCenter
                radius: guessLabel.radius
                currentSubLevel: 0
                onStop: Activity.nextSubLevel()

            }
            Rectangle {
                id: guessLabel
                anchors.top: parent.top
                anchors.bottom: parent.bottom
                anchors.right: parent.right
                anchors.left: score.right
                anchors.leftMargin: background.baseMargins
                radius: background.baseMargins
                border.color: "orange"
                border.width: 3 * ApplicationInfo.ratio
                color: "#E8E8E8"
                GCText {
                    id: guess
                    anchors.centerIn: parent
                    width: parent.width - 2 * background.baseMargins
                    height: parent.height - background.baseMargins
                    fontSizeMode: Text.Fit
                    minimumPointSize: 7
                    fontSize: mediumSize
                    verticalAlignment: Text.AlignVCenter
                    horizontalAlignment: Text.AlignHCenter
                    text: qsTr("Guesscount: %1").arg(items.result)
                }
            }
        }

        Column {
            id: col
            spacing: background.halfMargins
            anchors.top: top.bottom
            anchors.left: parent.left
            anchors.right: parent.right
            anchors.bottom: parent.bottom
            anchors.margins: background.baseMargins
            anchors.bottomMargin: bar.height * 1.2
            OperatorRow {
                id: operatorRow
                width: parent.width
                height: Math.min(70 * ApplicationInfo.ratio,
                                 parent.height / 6 - background.halfMargins)
                mode: items.mode
            }
            OperandRow {
                id: operandRow
                width: parent.width
                height: operatorRow.height
            }
            Repeater {
                id: repeat
                model: operatorRow.repeater.model.length
                delegate: OperationRow {
                    id: operationRow
                    width: col.width
                    height: operatorRow.height
                    property alias operationRow: operationRow
                    noOfRows: operatorRow.repeater.model.length
                    rowNo: modelData
                    guesscount: items.result
                    prevText: ""
                    prevComplete: prevText != "" ? true : false
                    reparent: items.clearOperations
                    onTextChanged: {
                        if(index + 1 < repeat.count) {
                            repeat.itemAt(index + 1).prevText = text
                        }
                    }
                }
            }
        }

        DialogHelp {
            id: dialogHelp
            onClose: home()
        }

        DialogChooseLevel {
            id: dialogActivityConfig
            currentActivity: activity.activityInfo
            onSaveData: {
                levelFolder = dialogActivityConfig.chosenLevels
                currentActivity.currentLevels = dialogActivityConfig.chosenLevels
                ApplicationSettings.setCurrentLevels(currentActivity.name, dialogActivityConfig.chosenLevels)
                activity.needRestart = true
            }
            onClose: {
                if(Activity.configDone(items.levelArr)){
                    Activity.initLevel()
                    home()
                }
            }
            onLoadData: {
                if(activityData && activityData["mode"]) {
                    items.mode = activityData["mode"]
                    if(activityData["levelArr"] === undefined) {
                        items.levelArr = activityData["levelArr"]

                    }
                    if(activityData["levelArr"]) {
                        items.levelArr = activityData["levelArr"]
                    }
                }
                else {
                    items.mode = "builtin"
                    items.levelArr = Activity.defaultOperators
                }
            }
            onStartActivity: {
                background.stop()
                background.start()
            }
        }

        Bar {
            id: bar
            level: items.currentLevel + 1
            content: BarEnumContent { value: help | home | level | activityConfig }
            onHelpClicked: {
                displayDialog(dialogHelp)
            }
            onPreviousLevelClicked: {
                Activity.previousLevel()
            }
            onNextLevelClicked: {
                Activity.nextLevel()
            }
            onActivityConfigClicked: {
                displayDialog(dialogActivityConfig)
            }
            onHomeClicked: activity.home()
        }

        Bonus {
            id: bonus
            Component.onCompleted: win.connect(Activity.nextLevel)
        }

        Rectangle {
            id: warningDialog
            width: Math.min(300 * ApplicationInfo.ratio, col.width)
            height: 70 * ApplicationInfo.ratio
            visible: false
            color: "#373737"
            radius: background.baseMargins
            border.color: "white"
            border.width: background.tileBorder
            property alias dialogText: dialogText
            anchors.centerIn: col
            GCText {
                id: dialogText
                anchors.centerIn: parent
                fontSize: regularSize
                fontSizeMode: Text.Fit
                color: "white"
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                width: parent.width - background.baseMargins
                height: parent.height - background.baseMargins * 2
                wrapMode: TextEdit.WordWrap
            }
            states: [
                State {
                    when: warningDialog.visible
                    PropertyChanges {
                        top {
                            opacity: 0.5
                        }
                    }
                    PropertyChanges {
                        col {
                            opacity: 0.5
                        }
                    }
                },
                State {
                    when: !warningDialog.visible
                    PropertyChanges {
                        top {
                            opacity: 1
                        }
                    }
                    PropertyChanges {
                        col {
                            opacity: 1
                        }
                    }
                }
            ]
        }
    }
}
