"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bucketEncryptionTransformer = bucketEncryptionTransformer;
exports.bucketVersioningTransformer = bucketVersioningTransformer;
exports.getBucketNotificationTransformer = getBucketNotificationTransformer;
exports.getBucketRegionTransformer = getBucketRegionTransformer;
exports.getCompleteMultipartTransformer = getCompleteMultipartTransformer;
exports.getConcater = getConcater;
exports.getCopyObjectTransformer = getCopyObjectTransformer;
exports.getErrorTransformer = getErrorTransformer;
exports.getHashSummer = getHashSummer;
exports.getInitiateMultipartTransformer = getInitiateMultipartTransformer;
exports.getListBucketTransformer = getListBucketTransformer;
exports.getListMultipartTransformer = getListMultipartTransformer;
exports.getListObjectsTransformer = getListObjectsTransformer;
exports.getListObjectsV2Transformer = getListObjectsV2Transformer;
exports.getListObjectsV2WithMetadataTransformer = getListObjectsV2WithMetadataTransformer;
exports.getListPartsTransformer = getListPartsTransformer;
exports.getNotificationTransformer = getNotificationTransformer;
exports.getTagsTransformer = getTagsTransformer;
exports.lifecycleTransformer = lifecycleTransformer;
exports.objectLegalHoldTransformer = objectLegalHoldTransformer;
exports.objectLockTransformer = objectLockTransformer;
exports.objectRetentionTransformer = objectRetentionTransformer;
exports.removeObjectsTransformer = removeObjectsTransformer;
exports.replicationConfigTransformer = replicationConfigTransformer;
exports.selectObjectContentTransformer = selectObjectContentTransformer;
exports.uploadPartTransformer = uploadPartTransformer;
var Crypto = _interopRequireWildcard(require("crypto"), true);
var _jsonStream = require("json-stream");
var _lodash = require("lodash");
var _through = require("through2");
var errors = _interopRequireWildcard(require("./errors.js"), true);
var _helper = require("./internal/helper.js");
var xmlParsers = _interopRequireWildcard(require("./xml-parsers.js"), true);
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * MinIO Javascript Library for Amazon S3 Compatible Cloud Storage, (C) 2015, 2016 MinIO, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// getConcater returns a stream that concatenates the input and emits
// the concatenated output when 'end' has reached. If an optional
// parser function is passed upon reaching the 'end' of the stream,
// `parser(concatenated_data)` will be emitted.
function getConcater(parser, emitError) {
  var objectMode = false;
  var bufs = [];
  if (parser && !(0, _helper.isFunction)(parser)) {
    throw new TypeError('parser should be of type "function"');
  }
  if (parser) {
    objectMode = true;
  }
  return _through({
    objectMode
  }, function (chunk, enc, cb) {
    bufs.push(chunk);
    cb();
  }, function (cb) {
    if (emitError) {
      cb(parser(Buffer.concat(bufs).toString()));
      // cb(e) would mean we have to emit 'end' by explicitly calling this.push(null)
      this.push(null);
      return;
    }
    if (bufs.length) {
      if (parser) {
        this.push(parser(Buffer.concat(bufs).toString()));
      } else {
        this.push(Buffer.concat(bufs));
      }
    }
    cb();
  });
}

// Generates an Error object depending on http statusCode and XML body
function getErrorTransformer(response) {
  var statusCode = response.statusCode;
  var code, message;
  if (statusCode === 301) {
    code = 'MovedPermanently';
    message = 'Moved Permanently';
  } else if (statusCode === 307) {
    code = 'TemporaryRedirect';
    message = 'Are you using the correct endpoint URL?';
  } else if (statusCode === 403) {
    code = 'AccessDenied';
    message = 'Valid and authorized credentials required';
  } else if (statusCode === 404) {
    code = 'NotFound';
    message = 'Not Found';
  } else if (statusCode === 405) {
    code = 'MethodNotAllowed';
    message = 'Method Not Allowed';
  } else if (statusCode === 501) {
    code = 'MethodNotAllowed';
    message = 'Method Not Allowed';
  } else {
    code = 'UnknownError';
    message = `${statusCode}`;
  }
  var headerInfo = {};
  // A value created by S3 compatible server that uniquely identifies
  // the request.
  headerInfo.amzRequestid = response.headersSent ? response.getHeader('x-amz-request-id') : null;
  // A special token that helps troubleshoot API replies and issues.
  headerInfo.amzId2 = response.headersSent ? response.getHeader('x-amz-id-2') : null;
  // Region where the bucket is located. This header is returned only
  // in HEAD bucket and ListObjects response.
  headerInfo.amzBucketRegion = response.headersSent ? response.getHeader('x-amz-bucket-region') : null;
  return getConcater(xmlString => {
    let getError = () => {
      // Message should be instantiated for each S3Errors.
      var e = new errors.S3Error(message);
      // S3 Error code.
      e.code = code;
      _lodash.each(headerInfo, (value, key) => {
        e[key] = value;
      });
      return e;
    };
    if (!xmlString) {
      return getError();
    }
    let e;
    try {
      e = xmlParsers.parseError(xmlString, headerInfo);
    } catch (ex) {
      return getError();
    }
    return e;
  }, true);
}

// A through stream that calculates md5sum and sha256sum
function getHashSummer(enableSHA256) {
  var md5 = Crypto.createHash('md5');
  var sha256 = Crypto.createHash('sha256');
  return _through.obj(function (chunk, enc, cb) {
    if (enableSHA256) {
      sha256.update(chunk);
    } else {
      md5.update(chunk);
    }
    cb();
  }, function (cb) {
    var md5sum = '';
    var sha256sum = '';
    if (enableSHA256) {
      sha256sum = sha256.digest('hex');
    } else {
      md5sum = md5.digest('base64');
    }
    var hashData = {
      md5sum,
      sha256sum
    };
    this.push(hashData);
    this.push(null);
    cb();
  });
}

// Following functions return a stream object that parses XML
// and emits suitable Javascript objects.

// Parses CopyObject response.
function getCopyObjectTransformer() {
  return getConcater(xmlParsers.parseCopyObject);
}

// Parses listBuckets response.
function getListBucketTransformer() {
  return getConcater(xmlParsers.parseListBucket);
}

// Parses listMultipartUploads response.
function getListMultipartTransformer() {
  return getConcater(xmlParsers.parseListMultipart);
}

// Parses listParts response.
function getListPartsTransformer() {
  return getConcater(xmlParsers.parseListParts);
}

// Parses initMultipartUpload response.
function getInitiateMultipartTransformer() {
  return getConcater(xmlParsers.parseInitiateMultipart);
}

// Parses listObjects response.
function getListObjectsTransformer() {
  return getConcater(xmlParsers.parseListObjects);
}

// Parses listObjects response.
function getListObjectsV2Transformer() {
  return getConcater(xmlParsers.parseListObjectsV2);
}

// Parses listObjects with metadata response.
function getListObjectsV2WithMetadataTransformer() {
  return getConcater(xmlParsers.parseListObjectsV2WithMetadata);
}

// Parses completeMultipartUpload response.
function getCompleteMultipartTransformer() {
  return getConcater(xmlParsers.parseCompleteMultipart);
}

// Parses getBucketLocation response.
function getBucketRegionTransformer() {
  return getConcater(xmlParsers.parseBucketRegion);
}

// Parses GET/SET BucketNotification response
function getBucketNotificationTransformer() {
  return getConcater(xmlParsers.parseBucketNotification);
}

// Parses a notification.
function getNotificationTransformer() {
  // This will parse and return each object.
  return new _jsonStream();
}
function bucketVersioningTransformer() {
  return getConcater(xmlParsers.parseBucketVersioningConfig);
}
function getTagsTransformer() {
  return getConcater(xmlParsers.parseTagging);
}
function lifecycleTransformer() {
  return getConcater(xmlParsers.parseLifecycleConfig);
}
function objectLockTransformer() {
  return getConcater(xmlParsers.parseObjectLockConfig);
}
function objectRetentionTransformer() {
  return getConcater(xmlParsers.parseObjectRetentionConfig);
}
function bucketEncryptionTransformer() {
  return getConcater(xmlParsers.parseBucketEncryptionConfig);
}
function replicationConfigTransformer() {
  return getConcater(xmlParsers.parseReplicationConfig);
}
function objectLegalHoldTransformer() {
  return getConcater(xmlParsers.parseObjectLegalHoldConfig);
}
function uploadPartTransformer() {
  return getConcater(xmlParsers.uploadPartParser);
}
function selectObjectContentTransformer() {
  return getConcater();
}
function removeObjectsTransformer() {
  return getConcater(xmlParsers.removeObjectsParser);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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