﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 */
class ModifyCustomDomainAssociationRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API ModifyCustomDomainAssociationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyCustomDomainAssociation"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The custom domain name for a changed custom domain association.</p>
   */
  inline const Aws::String& GetCustomDomainName() const { return m_customDomainName; }
  inline bool CustomDomainNameHasBeenSet() const { return m_customDomainNameHasBeenSet; }
  template <typename CustomDomainNameT = Aws::String>
  void SetCustomDomainName(CustomDomainNameT&& value) {
    m_customDomainNameHasBeenSet = true;
    m_customDomainName = std::forward<CustomDomainNameT>(value);
  }
  template <typename CustomDomainNameT = Aws::String>
  ModifyCustomDomainAssociationRequest& WithCustomDomainName(CustomDomainNameT&& value) {
    SetCustomDomainName(std::forward<CustomDomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The certificate Amazon Resource Name (ARN) for the changed custom domain
   * association.</p>
   */
  inline const Aws::String& GetCustomDomainCertificateArn() const { return m_customDomainCertificateArn; }
  inline bool CustomDomainCertificateArnHasBeenSet() const { return m_customDomainCertificateArnHasBeenSet; }
  template <typename CustomDomainCertificateArnT = Aws::String>
  void SetCustomDomainCertificateArn(CustomDomainCertificateArnT&& value) {
    m_customDomainCertificateArnHasBeenSet = true;
    m_customDomainCertificateArn = std::forward<CustomDomainCertificateArnT>(value);
  }
  template <typename CustomDomainCertificateArnT = Aws::String>
  ModifyCustomDomainAssociationRequest& WithCustomDomainCertificateArn(CustomDomainCertificateArnT&& value) {
    SetCustomDomainCertificateArn(std::forward<CustomDomainCertificateArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the cluster to change a custom domain association for.</p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  ModifyCustomDomainAssociationRequest& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_customDomainName;
  bool m_customDomainNameHasBeenSet = false;

  Aws::String m_customDomainCertificateArn;
  bool m_customDomainCertificateArnHasBeenSet = false;

  Aws::String m_clusterIdentifier;
  bool m_clusterIdentifierHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
