﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/resource-explorer-2/ResourceExplorer2Request.h>
#include <aws/resource-explorer-2/ResourceExplorer2_EXPORTS.h>
#include <aws/resource-explorer-2/model/SearchFilter.h>

#include <utility>

namespace Aws {
namespace ResourceExplorer2 {
namespace Model {

/**
 */
class ListResourcesRequest : public ResourceExplorer2Request {
 public:
  AWS_RESOURCEEXPLORER2_API ListResourcesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListResources"; }

  AWS_RESOURCEEXPLORER2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>An array of strings that specify which resources are included in the results
   * of queries made using this view. When you use this view in a <a>Search</a>
   * operation, the filter string is combined with the search's
   * <code>QueryString</code> parameter using a logical <code>AND</code>
   * operator.</p> <p>For information about the supported syntax, see <a
   * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/using-search-query-syntax.html">Search
   * query reference for Resource Explorer</a> in the <i>Amazon Web Services Resource
   * Explorer User Guide</i>.</p>  <p>This query string in the context of
   * this operation supports only <a
   * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/using-search-query-syntax.html#query-syntax-filters">filter
   * prefixes</a> with optional <a
   * href="https://docs.aws.amazon.com/resource-explorer/latest/userguide/using-search-query-syntax.html#query-syntax-operators">operators</a>.
   * It doesn't support free-form text. For example, the string <code>region:us*
   * service:ec2 -tag:stage=prod</code> includes all Amazon EC2 resources in any
   * Amazon Web Services Region that begins with the letters <code>us</code> and is
   * <i>not</i> tagged with a key <code>Stage</code> that has the value
   * <code>prod</code>.</p>
   */
  inline const SearchFilter& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = SearchFilter>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = SearchFilter>
  ListResourcesRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results that you want included on each page of the
   * response. If you do not include this parameter, it defaults to a value
   * appropriate to the operation. If additional items exist beyond those included in
   * the current response, the <code>NextToken</code> response element is present and
   * has a value (is not null). Include that value as the <code>NextToken</code>
   * request parameter in the next call to the operation to get the next part of the
   * results.</p>  <p>An API operation can return fewer results than the
   * maximum even when there are more results available. You should check
   * <code>NextToken</code> after every operation to ensure that you receive all of
   * the results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListResourcesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the Amazon resource name (ARN) of the view to use for the query. If
   * you don't specify a value for this parameter, then the operation automatically
   * uses the default view for the Amazon Web Services Region in which you called
   * this operation. If the Region either doesn't have a default view or if you don't
   * have permission to use the default view, then the operation fails with a 401
   * Unauthorized exception.</p>
   */
  inline const Aws::String& GetViewArn() const { return m_viewArn; }
  inline bool ViewArnHasBeenSet() const { return m_viewArnHasBeenSet; }
  template <typename ViewArnT = Aws::String>
  void SetViewArn(ViewArnT&& value) {
    m_viewArnHasBeenSet = true;
    m_viewArn = std::forward<ViewArnT>(value);
  }
  template <typename ViewArnT = Aws::String>
  ListResourcesRequest& WithViewArn(ViewArnT&& value) {
    SetViewArn(std::forward<ViewArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parameter for receiving additional results if you receive a
   * <code>NextToken</code> response in a previous request. A <code>NextToken</code>
   * response indicates that more output is available. Set this parameter to the
   * value of the previous call's <code>NextToken</code> response to indicate where
   * the output should continue from. The pagination tokens expire after 24
   * hours.</p>  <p>The <code>ListResources</code> operation does not generate
   * a <code>NextToken</code> if you set <code>MaxResults</code> to 1000. </p>
   *
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListResourcesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  SearchFilter m_filters;
  bool m_filtersHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_viewArn;
  bool m_viewArnHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResourceExplorer2
}  // namespace Aws
