﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/serverlessrepo/ServerlessApplicationRepositoryRequest.h>
#include <aws/serverlessrepo/ServerlessApplicationRepository_EXPORTS.h>
#include <aws/serverlessrepo/model/ApplicationPolicyStatement.h>

#include <utility>

namespace Aws {
namespace ServerlessApplicationRepository {
namespace Model {

/**
 */
class PutApplicationPolicyRequest : public ServerlessApplicationRepositoryRequest {
 public:
  AWS_SERVERLESSAPPLICATIONREPOSITORY_API PutApplicationPolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutApplicationPolicy"; }

  AWS_SERVERLESSAPPLICATIONREPOSITORY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the application.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  PutApplicationPolicyRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of policy statements applied to the application.</p>
   */
  inline const Aws::Vector<ApplicationPolicyStatement>& GetStatements() const { return m_statements; }
  inline bool StatementsHasBeenSet() const { return m_statementsHasBeenSet; }
  template <typename StatementsT = Aws::Vector<ApplicationPolicyStatement>>
  void SetStatements(StatementsT&& value) {
    m_statementsHasBeenSet = true;
    m_statements = std::forward<StatementsT>(value);
  }
  template <typename StatementsT = Aws::Vector<ApplicationPolicyStatement>>
  PutApplicationPolicyRequest& WithStatements(StatementsT&& value) {
    SetStatements(std::forward<StatementsT>(value));
    return *this;
  }
  template <typename StatementsT = ApplicationPolicyStatement>
  PutApplicationPolicyRequest& AddStatements(StatementsT&& value) {
    m_statementsHasBeenSet = true;
    m_statements.emplace_back(std::forward<StatementsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;

  Aws::Vector<ApplicationPolicyStatement> m_statements;
  bool m_applicationIdHasBeenSet = false;
  bool m_statementsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ServerlessApplicationRepository
}  // namespace Aws
