"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatHitProvider = formatHitProvider;
var _lodash = _interopRequireDefault(require("lodash"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const formattedCache = new WeakMap();
const partialFormattedCache = new WeakMap();

// Takes a hit, merges it with any stored/scripted fields, and with the metaFields
// returns a formatted version
function formatHitProvider(dataView, defaultFormat) {
  function convert(hit, val, fieldName, type) {
    const field = dataView.fields.getByName(fieldName);
    const format = field ? dataView.getFormatterForField(field) : defaultFormat;
    return format.convert(val, type, {
      field,
      hit,
      dataView
    });
  }
  function formatHit(hit, type = 'html') {
    // Cache is only used for formatType === 'html' (default)
    if (type === 'text') {
      const flattened = dataView.flattenHit(hit);
      const result = {};
      for (const [key, value] of Object.entries(flattened)) {
        result[key] = convert(hit, value, key, type);
      }
      return result;
    }
    const cached = formattedCache.get(hit);
    if (cached) {
      return cached;
    }

    // use and update the partial cache, but don't rewrite it.
    // _source is stored in partialFormattedCache but not formattedCache
    const partials = partialFormattedCache.get(hit) || {};
    partialFormattedCache.set(hit, partials);
    const cache = {};
    formattedCache.set(hit, cache);
    _lodash.default.forOwn(dataView.flattenHit(hit), function (val, fieldName) {
      // sync the formatted and partial cache
      if (!fieldName) {
        return;
      }
      const formatted = partials[fieldName] == null ? convert(hit, val, fieldName, type) : partials[fieldName];
      cache[fieldName] = partials[fieldName] = formatted;
    });
    return cache;
  }
  formatHit.formatField = function (hit, fieldName, type = 'html') {
    // Cache is only used for formatType === 'html' (default)
    if (type === 'html') {
      let partials = partialFormattedCache.get(hit);
      if (partials && partials[fieldName] != null) {
        return partials[fieldName];
      }
      if (!partials) {
        partials = {};
        partialFormattedCache.set(hit, partials);
      }
    }
    const val = fieldName === '_source' ? hit._source : dataView.flattenHit(hit)[fieldName];
    return convert(hit, val, fieldName, type);
  };
  return formatHit;
}