"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.docPermissionControlledSuite = docPermissionControlledSuite;
var _mocks = require("../../mocks");
var _utils = require("../../utils");
var utilsExports = _interopRequireWildcard(require("../../utils/auth_info"));
var _osd_server = require("../../../test_helpers/osd_server");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

function docPermissionControlledSuite() {
  const settings = {
    opensearchDashboards: {
      dashboardAdmin: {
        users: ['dashboard_admin']
      }
    },
    savedObjects: {
      permission: {
        enabled: true
      }
    },
    workspace: {
      enabled: true
    }
  };
  const startOpenSearchDashboards = async ({
    isDashboardAdmin
  }) => {
    const root = (0, _osd_server.createRootWithCorePlugins)(settings);
    const coreSetup = await root.setup();
    coreSetup.http.registerOnPostAuth((request, response, t) => {
      (0, _utils.updateWorkspaceState)(request, {
        isDashboardAdmin
      });
      return t.next();
    });
    const coreStart = await root.start();
    const osdServer = (0, _osd_server.getOsdServer)(root);
    return {
      root,
      osdServer,
      coreStart,
      stop: async () => await root.shutdown()
    };
  };
  const setup = async ({
    isDashboardAdmin
  }) => {
    const servers = (0, _osd_server.createTestServers)({
      adjustTimeout: t => jest.setTimeout(t),
      settings: {
        osd: settings
      }
    });
    const osd = await startOpenSearchDashboards({
      isDashboardAdmin
    });
    const opensearchServer = await servers.startOpenSearch();
    const savedObjectsClient = osd.coreStart.savedObjects.getScopedClient(_mocks.httpServerMock.createOpenSearchDashboardsRequest());
    const savedObjectsRepository = osd.coreStart.savedObjects.createInternalRepository();
    jest.spyOn(utilsExports, 'getPrincipalsFromRequest').mockImplementation(request => {
      if (isDashboardAdmin) {
        return {
          users: ['dashboard_admin']
        };
      }
      return {
        users: ['foo']
      };
    });
    return {
      opensearchServer,
      osd,
      osdServer: osd.osdServer,
      savedObjectsClient,
      savedObjectsRepository
    };
  };
  describe('permission controlled settings with dashboard admin', () => {
    let services;
    beforeAll(async () => {
      services = await setup({
        isDashboardAdmin: true
      });
    });
    afterEach(async () => {
      await services.savedObjectsRepository.delete('config', '_dashboard_admin');
    });
    afterAll(async () => {
      await services.opensearchServer.stop();
      await services.osd.stop();
    });
    it('create should be allowed when the user is dashboard admin', async () => {
      const {
        osdServer
      } = services;
      const {
        statusCode,
        result
      } = await osdServer.inject({
        method: 'POST',
        url: '/api/opensearch-dashboards/settings?scope=dashboard_admin',
        payload: {
          changes: {
            'securitySetting.test': 'test-value'
          }
        }
      });
      expect(statusCode).toBe(200);
      expect(result).toEqual({
        settings: {
          'securitySetting.test': {
            userValue: 'test-value'
          }
        }
      });
    });
    it('update should be allowed when the user is dashboard admin', async () => {
      const {
        osdServer,
        savedObjectsRepository
      } = services;
      await savedObjectsRepository.create('config', {
        'securitySetting.test': 'test-value'
      }, {
        id: '_dashboard_admin',
        permissions: {
          read: {
            users: ['*']
          }
        }
      });
      const {
        statusCode,
        result
      } = await osdServer.inject({
        method: 'POST',
        url: '/api/opensearch-dashboards/settings?scope=dashboard_admin',
        payload: {
          changes: {
            'securitySetting.test': 'admin-value'
          }
        }
      });
      expect(statusCode).toBe(200);
      expect(result).toEqual({
        settings: {
          'securitySetting.test': {
            userValue: 'admin-value'
          }
        }
      });
    });
    it('get should be allow allowed when the user is dashboard admin', async () => {
      const {
        osdServer,
        savedObjectsRepository
      } = services;
      await savedObjectsRepository.create('config', {
        'securitySetting.test': 'admin-value'
      }, {
        id: '_dashboard_admin',
        permissions: {
          read: {
            users: ['*']
          }
        }
      });
      const {
        statusCode,
        result
      } = await osdServer.inject({
        method: 'GET',
        url: '/api/opensearch-dashboards/settings'
      });
      expect(statusCode).toBe(200);
      expect(result).toEqual({
        settings: {
          buildNum: {
            userValue: expect.any(Number)
          },
          'securitySetting.test': {
            userValue: 'admin-value'
          }
        }
      });
    });
  });
  describe('permission controlled settings with non-dashboard admin', () => {
    let services;
    beforeAll(async () => {
      services = await setup({
        isDashboardAdmin: false
      });
    });
    afterAll(async () => {
      await services.opensearchServer.stop();
      await services.osd.stop();
    });
    it('create should returns 403 when the user is not dashboard admin', async () => {
      const {
        osdServer
      } = services;
      const {
        statusCode,
        result
      } = await osdServer.inject({
        method: 'POST',
        url: '/api/opensearch-dashboards/settings?scope=dashboard_admin',
        payload: {
          changes: {
            'securitySetting.test': 'test-value'
          }
        }
      });
      expect(statusCode).toBe(403);
      expect(result).toEqual({
        error: 'Forbidden',
        message: 'No permission for admin UI settings operations',
        statusCode: 403
      });
    });
    it('update should returns 403 when the user is not dashboard admin', async () => {
      const {
        osdServer,
        savedObjectsRepository
      } = services;
      await savedObjectsRepository.create('config', {
        'securitySetting.test': 'test-value'
      }, {
        id: '_dashboard_admin',
        permissions: {
          read: {
            users: ['*']
          }
        },
        overwrite: true
      });
      const {
        statusCode,
        result
      } = await osdServer.inject({
        method: 'POST',
        url: '/api/opensearch-dashboards/settings?scope=dashboard_admin',
        payload: {
          changes: {
            'securitySetting.test': 'test-value'
          }
        }
      });
      expect(statusCode).toBe(403);
      expect(result).toEqual({
        error: 'Forbidden',
        message: 'Invalid saved objects permission',
        statusCode: 403
      });
    });
    it('get should be allow allowed when the user is not dashboard admin', async () => {
      const {
        osdServer,
        savedObjectsRepository
      } = services;
      await savedObjectsRepository.create('config', {
        'securitySetting.test': 'test-value'
      }, {
        id: '_dashboard_admin',
        permissions: {
          read: {
            users: ['*']
          }
        },
        overwrite: true
      });
      const {
        statusCode,
        result
      } = await osdServer.inject({
        method: 'GET',
        url: '/api/opensearch-dashboards/settings'
      });
      expect(statusCode).toBe(200);
      expect(result).toEqual({
        settings: {
          buildNum: {
            userValue: expect.any(Number)
          },
          'securitySetting.test': {
            userValue: 'test-value'
          }
        }
      });
    });
  });
}